﻿"use strict";
/*
    ルビ文字と装飾タグのときについての補足として、実際の例を示す。
    {ル:<赤大:ル>ビ文字,るびもじ}に対して作成されるデータ構造は次のようになる。
    obj = {
        type: 'Ruby',
        value: [
            {
                type: 'Array',
                value; [
                    {
                        type: 'Tag',
                        value: [
                            {
                                type: Text,
                                value: "赤大"
                            },
                            {
                                type: Text,
                                value: "ル"
                            }
                        ]
                    },
                    {
                        type: 'Text',
                        value: "ビ文字"
                    }
                ]
            },
            {
                type: 'Array',
                value; [
                    {
                        type: 'Text',
                        value: "るびもじ"
                    }
                ]
            }
        ]
    }
 */
/**
 * AUCの掲示板に書き込む形式のテキストを解析して、IAucTextのツリー構造オブジェクトを作成する。
 * @param text
 * @returns
 */
function PerseAucText(text) {
    let Child = [];
    let first = true;
    // 最初に改行で区切る。
    text.split(/\n/).forEach(work => {
        // 先頭以外は改行に対するIAucTextを追加する。
        if (!first) {
            Child.push({ type: 'Bre', value: "" });
        }
        first = false;
        let text1 = "";
        while (work.length > 0) {
            // work の先頭から、"<", ">" "{"が見つかるまで抜き出す。
            // 抜き出した文字はtext1に追加し、残りをworkにセットする。
            const match = work.match(/^[^<>{]*/);
            if (match) {
                text1 += match[0];
                work = work.slice(match[0].length);
            }
            // workが空なら、ループを抜ける。
            if (work == "") {
                break;
            }
            // workの先頭が何かのタグの可能性があるので、解析する
            const anytag = PerseAucTextAnyTag(work);
            if (anytag) {
                if (text1) {
                    Child.push({ type: 'Text', value: text1 });
                    text1 = "";
                }
                Child.push(anytag.result);
                work = anytag.text;
                continue;
            }
            // ここまで来たら、workの先頭に"{"や"<"があってもタグでない。
            // workの１文字をtext1に移sし、解析処理を続ける
            text1 += work.slice(0, 1);
            work = work.slice(1);
        }
        // text1が空でないなら、テキストとして登録する
        if (text1) {
            Child.push({ type: 'Text', value: text1 });
            text1 = "";
        }
    });
    const obj = {
        type: 'Array',
        value: Child
    };
    return obj;
}
/**
 * textの先頭が何かのタグの可能性があるので、解析する
 *
 * タグがあれば解析結果を返す。
 *
 * なければundefinedを返す。
 * @param text
 * @returns
 */
function PerseAucTextAnyTag(text) {
    // ルビ文字タグ解析を
    const ruby = PerseAucTextRuby(text);
    if (ruby) {
        return ruby;
    }
    const link = PerseAucTextLink(text);
    if (link) {
        return link;
    }
    const dice = PerseAucTextDice(text);
    if (dice) {
        return dice;
    }
    const tag = PerseAucTextTag(text);
    if (tag) {
        return tag;
    }
    return undefined;
}
/**
 * textの先頭にルビ文字タグがあるとみなして、解析処理を行う。
 *
 * ルビ文字タグがある場合は、解析結果と残りの文字列を返す。
 * 適切なルビ文字タグが存在しない場合は、undefinedを返す。
 * @param text
 * @returns
 */
function PerseAucTextRuby(text) {
    let work = text;
    // workの先頭から"{ル:"までとそれ以外を取り分ける。
    const match = work.match(/^\{ル:/);
    if (!match) {
        return undefined;
    }
    work = work.slice(match[0].length);
    // workの先頭から最初の","にマッチする部分を抜き出す
    const match2 = work.match(/([^,]+),/);
    if (!match2) {
        return undefined;
    }
    const ruby = match2[1];
    work = work.slice(match2[0].length);
    // rubyに改行があるときは、ルビ文字タグではない
    if (ruby.indexOf("\n") >= 0) {
        return undefined;
    }
    //  workの先頭から"}"にマッチする部分を抜き出す
    const match3 = work.match(/([^}]+)\}/);
    if (!match3) {
        return undefined;
    }
    const rt = match3[1];
    work = work.slice(match3[0].length);
    // ここまでくれば、ruby, rtがルビ文字、workが残りの文字列
    // rubyとrtはさらに解析した値をセットする。
    const Child = [];
    Child.push(PerseAucText(ruby));
    Child.push(PerseAucText(rt));
    const obj = { type: 'Ruby', value: Child };
    return { result: obj, text: work };
}
/**
 * textの先頭にリンクタグがあるとみなして、解析処理を行う。
 *
 * リンクタグがある場合は、解析結果と残りの文字列を返す。
 * 適切なリンクタグが存在しない場合は、undefinedを返す。
 * @param text
 * @returns
 */
function PerseAucTextLink(text) {
    let work = text;
    // リンクタグは、次の４種類がある。
    // 1.同じ掲示板内リンク             >>{数字}
    // 2.他の掲示板へのリンク           <bbs:\d+(:\d+)?>
    // 3.部隊またはキャラへのリンク     <id:[0-9a-z]{4}(:\d)?>
    // 4.ギルドリンク                   <id:[AIMSW]\d{5}>
    //
    // なお、掲示板に表示される文字がそれぞれで微妙に違うので注意。
    // 1はマッチした文字が全部表示される。>>1 -> >>1
    // 2は<>を除いた部分が表示される。 <bbs:200001> -> bbs:200001
    // 3と4は、<id: >を除いた部分が表示される <id:1xkx:1> -> 1xkx:1
    // IAucTextにセットするときは、表示される文字部分のセットする。
    let link = "";
    if (work.slice(0, 1) == ">") {
        const match1 = work.match(/^(>>\d+)/);
        if (!match1) {
            return undefined;
        }
        link = match1[1];
        work = work.slice(match1[0].length);
    }
    else if (work.slice(0, 1) == "<") {
        const match2 = work.match(/^<(bbs:\d{6}(:\d+)?)>/);
        const match3 = work.match(/^<id:([0-9a-z]{4}(:[1-5])?|[AIMSW]\d{5})>/);
        if (match2) {
            link = match2[1];
            work = work.slice(match2[0].length);
        }
        else if (match3) {
            link = match3[1];
            work = work.slice(match3[0].length);
        }
        else {
            return undefined;
        }
    }
    else {
        return undefined;
    }
    const obj = { type: 'Link', value: link };
    return { result: obj, text: work };
}
/**
 * textの先頭にダイスタグがあるとみなして、解析処理を行う。
 *
 * ダイスタグがある場合は、解析結果と残りの文字列を返す。
 * 適切なダイスタグが存在しない場合は、undefinedを返す。
 * @param text
 * @returns
 */
function PerseAucTextDice(text) {
    let work = text;
    // AUC本来のダイスタグは、
    //<dice:{数字}d{数字}>または<dice:{数字}d>である。
    //
    // しかし、ツールが作成するダイスタグは、出目が決まっていることを考慮して、
    // <dice:{1～10の数字}}d(:{２桁の数字})+>
    // または
    // <dice:{1～10の数字}}d{ダイスの面数の数字}(:{２桁の数字})+>
    // となっている点に注意。
    // ダイスタグにマッチする文字列を探し、マッチする部分を抜き出す。
    const match = work.match(/^<dice:([1-9]|10)d(4|6|8|10|12|20)?(:\d\d)+>/);
    if (!match) {
        return undefined;
    }
    const dice = match[0];
    // マッチした長さ全体の分だけ、workから駆除。
    work = work.slice(match[0].length);
    const obj = { type: 'Dice', value: dice };
    return { result: obj, text: work };
}
/**
 * AUCの装飾タグに使われている文字列
 */
const AucTextTag = [
    "黒", "白", "赤", "緑", "青", "黄", "水", "紫", "橙", "茶", "灰", "桃", "虹",
    "大", "小", "消", "上", "下", "太", "斜",
];
/**
 * textの先頭に装飾タグがあるとみなして、解析処理を行う。
 *
 * 装飾タグがある場合は、解析結果と残りの文字列を返す。
 * 適切な装飾タグが存在しない場合は、undefinedを返す。
 * @param text
 * @returns
 */
function PerseAucTextTag(text) {
    let work = text;
    // 装飾タグの"<xxxx:"までを探す。
    const match1 = work.match(/^<([^:]+):/);
    if (!match1) {
        return undefined;
    }
    const tag = match1[1];
    work = work.slice(match1[0].length);
    // tagに、装飾種類を表す文字が入っていないときは、装飾タグではない。
    const re = new RegExp(`(${AucTextTag.join("|")})`);
    if (!tag.match(re)) {
        return undefined;
    }
    // 装飾タグを閉じる">"を探す。
    // ただし、<大:<id:1xkx>>のようなケースがあるので、単純に最初に見つかった">"ではだめ。
    // ">"が見つかったときに、それがリンクタグやダイスタグを閉じる">"であるかチェックする。
    let searchStart = 0;
    let innerText = "";
    while (searchStart < work.length) {
        const index = work.indexOf(">", searchStart);
        if (index == -1) {
            // > が見つからない。
            return undefined;
        }
        // いったん、">"までの文字を取り出す。
        innerText = work.slice(0, index);
        // ダイスタグやリンクタグを閉じる">"でないことのチェック
        // 装飾タグの中に装飾タグを入れるのは、AUC内でも禁止（正しく動作しない）ので、
        // このチェックの対象にしない。
        // 
        // innertextには">"が含まれていないので、マッチング文字列も最後に">"をつけてはダメ。
        // 付けると正しく動作しない。
        const match1 = innerText.match(/<dice:([1-9]|10)d(4|6|8|10|12|20)?(:\d\d)+$/);
        const match2 = innerText.match(/<(bbs:\d{6}(:\d+)?)$/);
        const match3 = innerText.match(/<id:([0-9a-z]{4}(:[1-5])?|[AIMSW]\d{5})$/);
        if (!match1 && !match2 && !match3) {
            // ダイスタグなどを閉じる">"ではなかったので、ループを抜ける。
            work = work.slice(index + 1);
            break;
        }
        // サーチ位置を１つずらして、続行。
        searchStart = index + 1;
    }
    const obj = PerseAucText(innerText);
    obj.type = 'Tag';
    obj.value.unshift({ type: 'Text', value: tag });
    return { result: obj, text: work };
}
/**
 * 画像ファイルのURLの共通部分を利用して、文字を変換するためのクラス
 *
 */
class BbsUrlConverter {
    constructor() {
        /**
         * urlの興津部分の文字（https://～image/upload）
         */
        this.path = "";
    }
    /**
     * pathを初期化する。
     */
    Init() {
        this.path = "";
    }
    /**
     * urlの共通部分を %AUC% に変換して、urlの文字を圧縮する。
     * @param url
     */
    Compress(url) {
        let match = url.match(/(.*image\/upload)\/(.*)/);
        if (match) {
            if (!this.path) {
                // pathが空ならセットし、置き換えた文字を返す。
                this.path = match[1];
                return `%AUC%/${match[2]}`;
            }
            else if (this.path == match[1]) {
                // path画からではなくても、matac[1]と一致するなら置き換えた文字を返す。
                return `%AUC%/${match[2]}`;
            }
        }
        // 置き換えができないので、元の文字を返す。
        return url;
    }
    /**
     * path中の %AUC% を置き換え、圧縮されたURLを戻す。
     * @param path
     * @returns
     */
    Expand(path) {
        return path.replace(/%AUC%/, this.path);
    }
    /**
     * docから、BbsImgPathのプロパティをセットする。
     * @param img
     */
    SetValueFromHtmElement(doc) {
        const auc_path = doc.querySelector(".auc_path");
        if (auc_path instanceof HTMLElement) {
            this.path = auc_path.dataset.value + "";
        }
        else {
            this.path = "";
        }
    }
    /**
     * HTML文字列を作成する。
     */
    ToHTML() {
        let html = ``;
        html = `<div class="auc_path" data-value="${this.path}">&nbsp;</div>`;
        return html;
    }
}
/**
 * Bbs情報をToHTML()でHTML文字列に変換するときのパラメータとなる値を保持するクラス
 *
 * プロパティは static である。
 */
class BbsParameter {
}
/**
 * 画像保存モードか否か
 */
BbsParameter.ImageSaveMode = false;
/**
 * アイコン画像を出力するか否か
 */
BbsParameter.IconImg = false;
/**
 * 添付画像を出力するか否か
 */
BbsParameter.AttachedImg = false;
/**
 * 部隊IDを出力するか否か
 */
BbsParameter.ForceId = false;
/**
 * 日付を出力するか否か
 */
BbsParameter.Date = false;
/**
 * 画像データを保存するフォルダ名
 */
BbsParameter.ImgFolder = "";
/**
 * 画像データのURL文字列を圧縮/展開するための変換器
 */
BbsParameter.UrlConverter = new BbsUrlConverter();
/**
 * 後述の class Bbs 内で使用するサブクラス
 * <img>の情報を保持する
 */
class BbsImg {
    constructor() {
        /**
         * ファイル名：ディレクトリ名やフォルダ名を取り除いた部分
         */
        this.file = "";
    }
    /**
     * imgから、BbsImgのプロパティをセットする。
     * @param img
     */
    SetValueFromHtmElement(img) {
        if (img.dataset.org) {
            // data-orgがあるなら保存されたHTMLファイルをペーストしていると判断する。
            // 保存されたHTMLでは org thum の URL が圧縮されているので、展開する。
            this.org = BbsParameter.UrlConverter.Expand(img.dataset.org);
            if (img.dataset.thum) {
                this.thum = BbsParameter.UrlConverter.Expand(img.dataset.thum);
            }
            this.file = img.src.replace(/.*[/\\]/, "");
        }
        else {
            // data-orgが無いならゲーム画面からペーストしていると判断する。
            // srcが縮小画像（サムネイル）ときは、本来の画像に変更する。
            // 本来の画像のファイル名が xxxx/yyyy.png のとき、
            // 縮小画像のファイル名がは、xxxx/thum/thum_yyyy.png 
            // のようになっていることを利用し、マッチングしたら置き換える。
            let src = img.src;
            const match = src.match(/(.*\/)thum\/thum_(.*)/);
            if (match) {
                this.thum = src;
                src = match[1] + match[2];
            }
            this.org = src;
            this.file = src.replace(/.*[/\\]/, "");
        }
    }
    /**
     * 画像が存在する（つまり、HTMLとして出力できる）ことをチェックする
     */
    IsValid() {
        return this.org ? true : false;
    }
    /**
     * HTML文字列を作る
     * @returns
     */
    ToHTML() {
        if (!this.org) {
            return "";
        }
        let attribute = `src="${this.org}"`;
        if (BbsParameter.ImageSaveMode) {
            attribute += ` data-src="${BbsParameter.ImgFolder}/${this.file}"`;
        }
        if (this.org) {
            const org = BbsParameter.UrlConverter.Compress(this.org);
            attribute += ` data-org="${org}"`;
        }
        if (this.thum) {
            const thum = BbsParameter.UrlConverter.Compress(this.thum);
            attribute += ` data-thum="${thum}"`;
        }
        return `<img ${attribute}>`;
    }
    /**
     * objから、BbsImgのプロパティをセットする。
     * @param obj
     */
    SetFromJson(obj) {
        Object.assign(this, obj);
    }
}
/**
 * アイコンのイメージ
 */
class BbsIconImg extends BbsImg {
    /**
     * コンストラクタ
     * @param Bbs
     */
    constructor(Bbs) {
        super();
        this.Bbs = Bbs;
    }
    SetValueFromHtmElement(img) {
        super.SetValueFromHtmElement(img);
    }
    /**
     * HTML文字列を作る
     * @returns
     */
    ToHTML() {
        let html = super.ToHTML();
        if (html) {
            // htmlは "<src …>"となっているはずなので、最後の">"以外の部分を取り出す。
            const tmp = html.slice(0, -1);
            let attribute = "";
            if (this.Bbs.Author) {
                attribute += ` title="${this.Bbs.Author}" alt="${this.Bbs.Author}"`;
            }
            attribute += ` width="96" height="96"`;
            html = `${tmp}${attribute}>`;
        }
        return html;
    }
    /**
     * JSON.perse()で使用するメソッド
     * @returns
     */
    toJSON() {
        // Bbsがあると循環参照になるので、Bbsだけundefinedにしたコピーを作って返す。
        let obj = { ...this };
        obj.Bbs = undefined;
        return obj;
    }
}
/**
 * 掲示板書き込み時に「画像」の項目をセットしたときに、作成される部分の情報を保持する。
 */
class BbsAttachedImg extends BbsImg {
    constructor() {
        /*
            掲示板書き込み時に「画像」の項目をセットしたときにゲーム画面からペーストしたHTMLには、
            次の２つのフォーマットのHTML記述が存在する
    
            フォーマット１：画像を通常表示する時
            <div class="free-img">
                <a class="bbs-image-popup">
                    <img src="XXX.jpg">
                </a>
            </div>
    
            フォーマット２：画像を右に９０度回転させて表示する時
            <div class="free-img">
                <a class="bbs-image-popup-r">
                    <div style="**途中省略**; transform:rotate(90deg);">
                        <img src="XXX.png">
                    </div>
                </a>
            </div>
    
            ２つの違いは、<a>要素のクラス名で判断する。
            
            -----
            
            また、本ツールを使って保存したHTMLファイル中では、
            <a>要素が削除され、中身が外に出され、次のようなフォーマットになる。
    
            フォーマット１Ａ：画像を通常表示する時
            <div class="free-img">
                <img src="XXX.jpg">
            </div>
    
            フォーマット２Ａ：画像を右に９０度回転させて表示する時
            <div class="free-img">
                <div class="rotate90">
                    <img src="XXX.png">
                </div>
            </div>
    
            <div class="select">内の<img>は代替spanに変換され、次のようなフォーマットになる。
            
            フォーマット１Ａ：　フォーマット１の代替フォーマット
            <div class="graph">
               // select以外の<div>要素は、ゲーム画面をコピペ時と同じなので省略
                <div class="select">
                    <span class="enquete-img"></span>
                    項目１
                </div>
            </div>
            
            フォーマット１Ａとフォーマット２Ａについては、ToHTML()で作成される文字列によって変化するので、
            ToHTML()の方も参照して欲しい。
        */
        super(...arguments);
        /**
         * 画像の回転があるか否か
         *
         * 画像を通常表示するならfalse、90度回転させて表示するならtrueになる。
         */
        this.rotate = false;
    }
    /**
     * HTMLElementの内部を検索して、BbsAttachedImgのプロパティをセットする。
     * @param element
     * @returns
     */
    SetValueFromHtmElement(element) {
        // a要素を検索する。
        const a = element.querySelector('.free-img a.bbs-image-popup-r, .free-img a.bbs-image-popup');
        if (a instanceof HTMLAnchorElement) {
            // a要素が見つかったときは、フォーマット１またはフォーマット２である。
            // bbs-image-popup-rがクラスにあれば、右回転している。
            if (a.classList.contains("bbs-image-popup-r")) {
                this.rotate = true;
            }
            else {
                this.rotate = false;
            }
            // <a>の内部の<img>からsrcなどの情報をセットする
            const img = a.querySelector('img');
            if (img instanceof HTMLImageElement) {
                super.SetValueFromHtmElement(img);
            }
        }
        else {
            // a要素が見つかないので、フォーマット１Ａまたはフォーマット２Ａの考えて、情報をセットする
            let imgSelector = ""; // imgを検索するときのセレクタ
            // <div class="rotate90">を検索する。
            // rotate90があれな、右回転している。
            const div_rotate90 = element.querySelector('.free-img>div.rotate90');
            if (div_rotate90 instanceof HTMLDivElement) {
                this.rotate = true;
                imgSelector = ".free-img>div.rotate90>img";
            }
            else {
                this.rotate = false;
                imgSelector = ".free-img>img";
            }
            // imgを検索する。
            const img = element.querySelector(imgSelector);
            if (img instanceof HTMLImageElement) {
                super.SetValueFromHtmElement(img);
            }
        }
    }
    /**
     * HTML文字列を作る。
     * @returns
     */
    ToHTML() {
        let html = super.ToHTML();
        if (html) {
            if (this.rotate) {
                html = `<div class="rotate90">${html}</div>`;
            }
            html = `<div class="free-img">${html}</div>`;
        }
        return html;
    }
}
/**
 * さいころ１個分の情報を保持するクラス
 */
class BbsDice {
    constructor() {
        /**
         * ダイスのタイプ
         */
        this.type = "";
        /**
         * 出目
         */
        this.value = "";
    }
    /*
        ゲーム画面からコピペしたHTMLでは、
        １つのダイスに対して次のような<img>要素が出力される。

        (1) ダイスがさいころの時
        <img src="(重要部分以外省略)/sai_05.png">

        (2) ダイスが D4 や D20 などの時
        <img src="(重要部分以外省略)/d20_05.png">

        d20の部分はn面体ダイスなら`d{$n}`になる。

        そこで、elementがimgなら img.src を解析して、ダイスタイプと出目をセットする
        
        -----

        本ツールを使って保存したHTMLからコピペした場合は、
        ファイル中のダイスは代替spanに変換されている。
        それが次のようなフォーマットになっている。

        <span class="sai" data-dicetype="sai" data-value="05"></span>
    */
    /**
     * elementのから、BbsDiceのプロパティをセットする。
     * 正しくセットで来たときは trueを返し、そうでないときは falseを返す。
     * @param element
     */
    SetValueFromHTMLElement(element) {
        if (element instanceof HTMLImageElement) {
            // elmentが<img>ならsrcを解析してプロパティをセットする
            const match = element.src.match(/dice\/(sai|d4|d6|d8|d10|d12|d20)_([01][0-9])\.png/);
            if (match) {
                this.type = match[1]; // ダイスタイプ
                this.value = match[2]; // 出目
                return true;
            }
        }
        else {
            // elmentが<img>でないので代替spanと考え、datasetからプロパティをセットする
            const diceType = element.dataset.dicetype;
            const value = element.dataset.value;
            if (diceType && value) {
                this.type = diceType;
                this.value = value;
                return true;
            }
        }
        // プロパティが正しくセットできていないので、false を返す。
        return false;
    }
    /**
     *  HTML文字列を作る。
     * @returns
     */
    ToSring() {
        let diceType = this.type;
        let value = this.value;
        if (diceType && value) {
            switch (diceType) {
                case 'sai':
                    // さいころは出目を１桁にする
                    value = value.slice(-1);
                    // 出目が１のときだけ、値を変更する
                    if (value == '1') {
                        value = '●';
                    }
                    break;
                case 'd4':
                case 'd6':
                case 'd8':
                case 'd10':
                    // ４面、６面、８面、１０面のダイスは出目を１桁にする
                    value = value.slice(-1);
                    break;
            }
            return `<span class="sai" data-dicetype="${diceType}" data-value="${this.value}">${value}</span>`;
        }
        else {
            return "";
        }
    }
}
/**
 * BbsDiceを複数持つクラスの基本となるクラス
 *
 */
class BbsDiceArray {
    constructor() {
        /**
         * ダイスのリスト
         */
        this.Dices = [];
    }
    /**
     * diceを追加する。
     * @param dice
     */
    AddDice(dice) {
        this.Dices.push(dice);
    }
    /**
     * targetが保持するダイスタイプが一致するなら、targetから取り出して、thisに格納する。
     * @param target
     */
    Merge(target) {
        const diceType = this.Dices[0].type;
        while (target.Dices.length > 0) {
            const dice = target.Dices.shift();
            if (!dice) {
                break;
            }
            if (dice.type != diceType) {
                target.Dices.unshift(dice);
                break;
            }
            this.Dices.push(dice);
        }
    }
    /**
     * ダイスが有効な（つまり、HTMLとして出力できる）ことをチェックする
     */
    IsValid() {
        // ダイスが１つもないなら無効。１つ以上あるなら有効。
        return (this.Dices.length > 0);
    }
    /**
     * AUCに書き込む時と同じような文字列を作成する。
     * @returns
     */
    ToAucString() {
        // 本来、掲示板書き込み時に「ダイス」の項目をセットしたときに表示されるダイスなので、
        // 「AUCに書き込む時」というのはちょっと違うのだが、テキスト中に表示するダイスと同様
        // の文字列を作って返す。
        const diceType = this.Dices[0].type;
        let diceValue = this.Dices.map(dice => dice.value).join(":");
        if (diceType == "sai") {
            return `<dice:${this.Dices.length}d:${diceValue}>`;
        }
        else {
            return `<dice:${this.Dices.length}${diceType}:${diceValue}>`;
        }
    }
    /**
     * ToAucString()で作った文字列から、BbsAttachedDiceのプロパティをセットする。
     * @param text
     */
    SetFromAucString(text) {
        this.Dices = [];
        const match = text.match(/^<dice:([1-9]|10)(d(4|6|8|10|12|20)?)((:\d\d)+)>$/);
        if (match) {
            // match[2]は"d"とか"d10"のようになっている。
            let diceType = match[2];
            if (diceType == "d") {
                diceType = "sai";
            }
            // match[4]は ":01:04:02"のようになっているので、先頭の":"を取り除く。
            const values = match[4].slice(1);
            // そして"01:04:02"としたあと、":"で分割 -> BbsDice()に変換 -> 登録。
            values.split(/:/).forEach(value => {
                const dice = new BbsDice();
                dice.type = diceType;
                dice.value = value;
                this.AddDice(dice);
            });
        }
    }
    /**
     * JSON.perse()で使用するメソッド
     *
     * ダイス情報は、ToAucString()で変換した文字列にする。
     * @returns
     */
    toJSON() {
        return this.ToAucString();
    }
    /**
     * JSON,parseで作られたobjectからBbsDiceArrayのプロパティをセットする。
     * @param text
     */
    SetFromJson(text) {
        this.SetFromAucString(text);
    }
}
/**
 * 掲示板書き込み時に「ダイス」の項目をセットしたときに、作成される部分の情報を保持する。
 */
class BbsAttachedDice extends BbsDiceArray {
    /*
        掲示板に書き込む時に「ダイス」の項目をセットすると、次のようなHTMLが作成される。
        
        <div class="free-img">
            <p class="dice">
                <img src="xxx.png">&nbsp;
                <img src="xxx.png">&nbsp;
                <img src="xxx.png">&nbsp;
                // 上記のimgのsrcの詳細は、 class BbsDice で説明している。
            </p>
        </div>
        
        imgはダイスを表示するための画像である。
        振ったダイスの数だけ存在する。（１～３個）
        
        また、本ツールを使って保存したHTMLファイル中では、
        ダイスは代替spanに変換され、次のようなHTMLが作成される。
        <div class="free-img">
            <p class="dice">
                <span class="sai"></span>&nbsp;
                <span class="sai"></span>&nbsp;
                <span class="sai"></span>&nbsp;
                // 上記のimgの代替sapn詳細は、 class BbsDice で説明している。
            </p>
        </div>
        
        どちらのケースも、<p class="dice">の内部にダイス要素が存在している。
        これを利用してダイスの値を認識する。
     */
    /**
     * ementの内部を検索して、BbsAttachedDiceのプロパティをセットする。
     * @param element
     * @returns
     */
    SetValueFromHtmElement(element) {
        // <div class="free-img">の内部の<p class="dice"を探す
        const p_dice = element.querySelector('.free-img p.dice');
        if (p_dice instanceof HTMLParagraphElement) {
            for (const child of p_dice.querySelectorAll('img, span.sai')) {
                if (child instanceof HTMLElement) {
                    const dice = new BbsDice();
                    const isValid = dice.SetValueFromHTMLElement(child);
                    if (isValid) {
                        this.AddDice(dice);
                    }
                }
            }
        }
    }
    /**
     * HTML文字列を作る。
     * @returns
     */
    ToHTML() {
        let html = '';
        if (this.Dices.length > 0) {
            for (const dice of this.Dices) {
                html += dice.ToSring() + '&nbsp;';
            }
            html = `<p class="dice">${html}</p>`;
            html = `<div class="free-img">${html}</div>`;
        }
        return html;
    }
}
/**
 * アンケートの項目１つ分の情報を保持する
 *
 * アンケートを設定したときの「項目」１つ分の情報を保持する。
 *
 * 掲示板中のHTMLでは、 \<div class="graph"> の部分に相当する。
 */
class BbsEnqueteItem {
    constructor() {
        /**
         * 選択肢
         *
         * 掲示板中のHTMLでは、 \<div class="select"> の部分に相当する。
         */
        this.Selection = "";
    }
    /*
        ゲーム画面をコピペした時
        アンケートの項目には２種類のフォーマットが存在する。
        
        フォーマット１：すでに投票済み or 投票期間が終了しているケース
        <div class="graph">
            <div class="percent-base"></div>
            <div class="percent" style="width:20%;"></div>
            <div class="percent-num">20％</div>
            <div class="select">
                <img src="https://han.au-chronicle.jp/web/img/bbs_enquete_vote.png">
                項目１
            </div>
        </div>
        
        フォーマット２：投票期間中だが、まだ投票していないケース
        <div class="graph">
            <div class="btn-line"><button>項目１</button></div>
        </div>
        
        -----
        
        また、本ツールを使って保存したHTMLファイル中では、
        <div class="select">内の<img>は代替spanに変換され、次のようなフォーマットになる。
        
        フォーマット１Ａ：　フォーマット１の代替フォーマット
        <div class="graph">
           // select以外の<div>要素は、ゲーム画面をコピペ時と同じなので省略
            <div class="select">
                <span class="enquete-img"></span>
                項目１
            </div>
        </div>
        
        同様に、
        <div class="btn-line">内の<button>は代替spanに変換され、次のようなフォーマットになる。
        
        フォーマット２Ａ：　フォーマット２の代替フォーマット
        <div class="graph">
            <div class="btn-line"><span class="enquete-button">項目１</span></div>
        </div>
        
        フォーマット１Ａとフォーマット２Ａについては、ToHTML()で作成される文字列によって変化するので、
        ToHTML()の方も参照して欲しい。
    */
    /**
     * div_graph の内部を検索して、BbsEnqueteItem のプロパティをセットする。
     * @param div_graph
     * @returns
     */
    SetValueFromHtmElement(div_graph) {
        // フォーマット２のbuttonまたはフォーマット２Ａのspanを探す 
        const button = div_graph.querySelector(".btn-line button, .btn-line span.enquete-button");
        if (button instanceof HTMLElement) {
            // 選択肢をセット
            this.Selection = button.innerText;
        }
        else {
            // ここに来たということは、フォーマット２や２Ａではなかったことになるので、
            // フォーマット１または１Ａとして解析を行なっていく、
            // 投票の比率をセット
            const percent_num = div_graph.querySelector(".percent-num");
            if (percent_num instanceof HTMLDivElement) {
                // 投票の比率は、"10％"や"83.3％"などになっている。
                // 取り扱いやすくするために数字だけをプロパティにセットする。
                const match = percent_num.innerText.match(/(\d+(\.\d+)?)％/);
                if (match) {
                    this.VotingRatio = match[1];
                }
            }
            // 選択肢とVoteをセット
            const select = div_graph.querySelector(".select");
            if (select instanceof HTMLDivElement) {
                // 選択肢をセット
                // 単純に、select.innerText で取得すると、
                // 子要素に<img>や<span>が入っているときに正しい「選択肢」が取れない。
                // そこで、純粋にテキスト要素だけを連結した文字を選択肢とする。
                let selection = '';
                for (const child of select.childNodes) {
                    if (child.nodeType == Node.TEXT_NODE) {
                        selection += child.nodeValue;
                    }
                }
                this.Selection = selection;
                // Voteをセット
                // フォーマット１の場合のimgを探す。
                const select_img = select.querySelector("img");
                if (select_img instanceof HTMLImageElement) {
                    // 投票している時は、
                    // <img src="(重要部分以外省略)/bbs_enquete_vote.png">
                    // そこで、srcの最後の８文字が"vote.png"なら投票したと判定する。
                    if (select_img.src.slice(-8) == "vote.png") {
                        this.Vote = true;
                    }
                    else {
                        this.Vote = false;
                    }
                }
                else {
                    // select_imgがない場合は、フォーマット１Ａとして解析する。
                    // 投票しているとき、代替spanは、下のようになる。
                    // <span class="enquete-img" data-type="vote">★</span>
                    // そこで、data-typeが"vote"なら投票したと判定する。
                    const select_img_span = select.querySelector("span.enquete-img");
                    if (select_img_span instanceof HTMLSpanElement) {
                        if (select_img_span.dataset.type == "vote") {
                            this.Vote = true;
                        }
                        else {
                            this.Vote = false;
                        }
                    }
                }
            }
            // 読み取れないデータがあったときは、falseを返す。
            if (this.VotingRatio == undefined || this.Vote == undefined) {
                return false;
            }
        }
        return true;
    }
    /**
     * 投票の比率に対応する数値を返す。
     */
    get VotingRatioNumber() {
        if (this.VotingRatio) {
            return Number(this.VotingRatio);
        }
        // 適切な数値を返せないときは、-1（投票比率として有り得ない値）を返す。
        return -1;
    }
    /**
     * HTML文字列を作る。
     *
     * @param maxVoing trueのとき、投票の比率が最大であるケースのHTMLを作成する
     * @returns
     */
    ToHTML(maxVoing = false) {
        let html = '';
        if (this.Vote == undefined) {
            // 投票済みでないケース
            //  <div class="btn-line"><button>項目１</button></div>ではなく、
            // 代替spanにする。
            html += `<span class="enquete-button">${this.Selection}</span>`;
            html = `<div class="btn-line">${html}</div>`;
        }
        else {
            // 投票済みのケース
            html += `<div class="percent-base"></div>`;
            if (this.VotingRatio != "0") {
                // 比率が0のときは、出力しない
                const maxVoingClass = maxVoing ? " percent-top" : "";
                html += `<div class="percent${maxVoingClass}" style="width: ${this.VotingRatio}%;"></div>`;
            }
            html += `<div class="percent-num">${this.VotingRatio}％</div>`;
            let div_select = '';
            // 
            if (this.Vote) {
                // 代替spanにする。
                div_select = `<span class="enquete-img" data-type="vote">★</span>`;
            }
            else {
                // 上記同様、代替spanにする。
                div_select = `<span class="enquete-img" data-type="sel">・</span>`;
            }
            div_select = `<div class="select">${div_select}${this.Selection}</div>`;
            html += div_select;
        }
        html = `<div class="graph">${html}</div>`;
        return html;
    }
    /**
     * objから、BbsEnqueteItem のプロパティをセットする。
     * @param obj
     */
    SetFromJson(obj) {
        Object.assign(this, obj);
    }
}
/**
 * 掲示板書き込み時に「アンケート」の項目をセットしたときに、作成される部分の情報を保持する。
 */
class BbsEnquete {
    constructor() {
        /**
         * アンケートの設問
         */
        this.Question = "";
        /**
         * アンケートの項目
         */
        this.EnquateItems = [];
        /**
         * 全項目の中の最大投票比率
         */
        this._MaxVotingRatio = -1;
    }
    // アンケートに対して、以下のようなでHTMLが出力される。
    //
    // <div class="enquete">
    //     <p class="wordBreak">アンケートの設問</p>
    //     <div class="graph"></div> 
    //     <div class="graph"></div>
    //     <div class="graph"></div>
    //     <div class="limit">投票期間は終了しました または 投票終了までの残り日時 </div>
    // </div>
    // 
    // <div class="graph">には、アンケートの項目数に応じて増減する。
    // また、内部にさらにHTMLがあるが、ここでは省略している。
    // 
    /**
     * HTMLElementの内部を検索して、BbsEnqueteのプロパティをセットする。
     * @param element
     * @returns
     */
    SetValueFromHtmElement(element) {
        // enquateが無ければ、アンケートがないので、即座に終了。
        const enquete = element.querySelector('.enquete');
        if (!enquete) {
            return;
        }
        // 設問をセットする
        const p_wordBreak = enquete.querySelector('p.wordBreak');
        if (p_wordBreak instanceof HTMLParagraphElement) {
            this.Question = p_wordBreak.innerText;
        }
        // 項目をセットする
        // 同時に、最大投票数をセットする。
        this._MaxVotingRatio = -1;
        for (const graph of enquete.querySelectorAll('div.graph')) {
            if (graph instanceof HTMLDivElement) {
                const item = new BbsEnqueteItem();
                const IsValid = item.SetValueFromHtmElement(graph);
                if (IsValid) {
                    this.EnquateItems.push(item);
                    const ratio = item.VotingRatioNumber;
                    if (this._MaxVotingRatio < ratio) {
                        this._MaxVotingRatio = ratio;
                    }
                }
            }
        }
    }
    /**
     * アンケートが有効な（つまり、HTMLとして出力できる）ことをチェックする
     */
    IsValid() {
        // アンケートの項目が１つもないなら無効。１つ以上あるなら有効。
        return (this.EnquateItems.length > 0);
    }
    /**
     * HTML文字列を作る。
     * @returns
     */
    ToHTML() {
        let html = '';
        if (this.EnquateItems.length > 0) {
            html += `<p class="wordBreak">${this.Question}</p>`;
            for (const item of this.EnquateItems) {
                const maxVoting = (item.VotingRatioNumber == this._MaxVotingRatio);
                html += item.ToHTML(maxVoting);
            }
            html = `<div class="enquete">${html}</div>`;
        }
        return html;
    }
    ToAucString() {
        let text = "";
        text += `アンケート：${this.Question}\n`;
        for (const item of this.EnquateItems) {
            text += `項目:${item.Selection}:${item.VotingRatio}%\n`;
        }
        return text;
    }
    /**
     * objから、BbsEnquete のプロパティをセットする。
     * @param obj
     */
    SetFromJson(obj) {
        Object.assign(this, obj);
    }
}
/**
 * 掲示板の本文部分を構成するテキスト
 */
class BbsTextText {
    constructor() {
        /**
         * テキスト
         */
        this.Text = "";
    }
    ToHTML() {
        let html = this.Text
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/ /g, '&nbsp;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;');
        return html;
    }
    ToAucString() {
        return this.Text;
    }
}
/**
 * 掲示板の本体部分のを構成する改行（<bre>）
 */
class BbsTextBre {
    ToHTML() {
        return "<br>";
    }
    ToAucString() {
        return "\n";
    }
}
/**
 * 掲示板の本体部分のを構成するリンク要素（<a>）
 */
class BbsTextAnchor extends BbsTextText {
    constructor() {
        super(...arguments);
        /**
         * class="bbs-mes-link"が指定されているか否か
         */
        this.BbsMesLink = false;
    }
    ToHTML() {
        let html = super.ToHTML();
        let className = "forceid";
        if (this.BbsMesLink) {
            className += " bbs-mes-link";
        }
        const resNo = this._GetInnerBbsResNo();
        if (resNo) {
            html = `<a class="${className}" href="#${Bbs.IdNameForResNo(resNo)}">${html}</a>`;
        }
        else {
            html = `<span class="${className}">${html}</span>`;
        }
        return html;
    }
    /**
     * 同じ掲示板内の記事へのリンクか否かをチェックする。
     *
     * 同じ掲示板内の記事へのリンクのときは、その記事番号を返す。
     * ちがうときはundefinedを返す。
     * @returns
     */
    _GetInnerBbsResNo() {
        // Textが">>数字"のときは、同じ掲示板内へのリンクと判断する。
        const match = this.Text.match(/^>>(\d+)$/);
        if (match) {
            return match[1];
        }
    }
    ToAucString() {
        let text = super.ToAucString();
        const match1 = text.match(/^bbs:\d+(:\d+)?$/); // 掲示板へのリンク
        const match2 = text.match(/^[0-9a-z]{4}(:\d)?$/); // 部隊 or キャラへのリンク
        const match3 = text.match(/^[AIMSW]\d{5}$/); // ギルドへのリンク
        // 同じ掲示板内へのリンク(>>数字)はtextにそのまま入っていて、変換の必要が無いのでNo check
        if (match1) {
            // これにマッチする場合は、掲示板へのリンク
            text = `<${text}>`;
        }
        else if (match2 || match3) {
            // これにマッチする場合は、部隊 or キャラ or ギルドへのリンク
            text = `<id:${text}>`;
        }
        return text;
    }
}
/**
 * 掲示板の本体部分を構成するサイコロ（ダイス）
 */
class BbsTextDice extends BbsDiceArray {
    ToHTML() {
        let html = "";
        for (const dice of this.Dices) {
            html += dice.ToSring();
        }
        return html;
    }
}
/**
 * 他のIBbsTextElementを子要素に持つ掲示板の本体部分
 */
class BbsTextParent {
    constructor() {
        /*
            ※タグの中にはルビを書くことができる。例として次のような書き込みができる。
            <赤大:タグの中にルビ文字の例{ル:赤大,あかだい}>
            
            これに対するHTML記述は、
            <span>タグの中にルビ文字<ruby>赤大<rt>あかだい</rt></ruby></span>  (styleは省略)
            のようになり、単純なテキストにできず、子要素を格納する必要がある。
    
            本クラスはそういったケースに対応するための汎用的なクラス
         */
        /**
         * 子要素
         */
        this.ChildElement = [];
    }
    /**
     * nodeの内部を検索して、BbsTextParentのChildElementをセットする。
     * @param span
     */
    SetValueFromHtmElement(node) {
        // 子要素
        for (const child of node.childNodes) {
            let textElement = CreateBbsTextElementFromNode(child);
            if (textElement instanceof BbsTextDice) {
                // ダイスのときは既存のダイスとマージできるか試す。
                const lastElement = this.ChildElement[this.ChildElement.length - 1];
                if (lastElement instanceof BbsTextDice) {
                    lastElement.Merge(textElement);
                    if (!textElement.IsValid()) {
                        textElement = null;
                    }
                }
            }
            if (textElement) {
                this.ChildElement.push(textElement);
            }
        }
    }
    ToHTML() {
        // 子要素
        let html = "";
        for (const child of this.ChildElement) {
            html += child.ToHTML();
        }
        return html;
    }
    ToAucString() {
        let text = "";
        for (const child of this.ChildElement) {
            text += child.ToAucString();
        }
        return text;
    }
    /**
     * ToAucString()で作った文字列から、BbsTextParentのChildElementをセットする。
     * @param text
     */
    SetFromAucString(text) {
        this.ChildElement = [];
        const obj = PerseAucText(text);
        this.SetFromIAucText(obj);
    }
    /**
     * ToAucString()で作った文字列を、PerseAucText()で変換したデータ構造から
     * ChildElementをセットする。
     * @param obj
     */
    SetFromIAucText(obj) {
        const elements = obj.value;
        elements.forEach(element => {
            const value = element.value;
            switch (element.type) {
                case 'Text':
                    const text = new BbsTextText();
                    text.Text = value;
                    this.ChildElement.push(text);
                    break;
                case 'Bre':
                    this.ChildElement.push(new BbsTextBre());
                    break;
                case 'Dice':
                    const dice = new BbsTextDice();
                    dice.SetFromAucString(value);
                    this.ChildElement.push(dice);
                    break;
                case 'Link':
                    const a = new BbsTextAnchor();
                    a.Text = value;
                    // 文字列が"<id:"で始まっているなら、部隊やキャラやギルドへのリンク。
                    // 始まっていないなら、">>1"や"<bbs:"のように掲示板へのリンク。
                    // 掲示板へのリンクのときは、BbsMesLinkをtrueにする。
                    a.BbsMesLink = (a.Text.match(/<id:/) == null);
                    this.ChildElement.push(a);
                    break;
                case 'Ruby':
                    const child = element.value;
                    const ruby = new BbsTextRuby();
                    ruby.SetFromIAucText(child[0]);
                    ruby.Rt.SetFromIAucText(child[1]);
                    this.ChildElement.push(ruby);
                    break;
                case 'Tag':
                    const tag = new BbsTextTag();
                    tag.SetFromIAucText(element);
                    this.ChildElement.push(tag);
                    break;
            }
        });
    }
}
/**
 * 装飾タグ
 */
class BbsTextTag extends BbsTextParent {
    constructor() {
        super(...arguments);
        /**
         * 太字（太）
         */
        this.Bold = false;
        /**
         * イタリック（斜）
         */
        this.Italic = false;
    }
    /**
     * spanの内部を検索して、BbsTextTagのプロパティをセットする。
     * @param span
     */
    SetValueFromHtmElement(span) {
        // 継承クラスのメソッドを使う事で、子要素はセットできる。
        super.SetValueFromHtmElement(span);
        // 太
        if (span.classList.contains("b")) {
            this.Bold = true;
        }
        // 斜
        if (span.classList.contains("i")) {
            this.Italic = true;
        }
        // 消上下
        for (const decoration in BbsTextTag.TextDecorationMap) {
            if (span.classList.contains(decoration)) {
                this.TextDecoration = decoration;
                break;
            }
        }
        // 大小
        // 大小はクラス指定が無いのでstyleから入力
        for (const key in BbsTextTag.FontSizeMap) {
            if (span.style.fontSize == BbsTextTag.FontSizeMap[key].value) {
                this.FontSize = key;
                break;
            }
        }
        // 色
        if (span.classList.contains("font_rainbow")) {
            // 虹
            this.Color = 'font_rainbow';
        }
        else {
            // 虹以外の色はクラス指定が無いので、styleから入力
            const colorString = Color(span.style.color);
            for (const key in BbsTextTag.ColorMap) {
                if (colorString == BbsTextTag.ColorMap[key].value) {
                    this.Color = key;
                    break;
                }
            }
        }
    }
    /**
     * タグとして意味のある情報がセットされているか否か
     */
    IsValid() {
        return (this.Bold ||
            this.Italic ||
            this.TextDecoration !== undefined ||
            this.Color !== undefined ||
            this.FontSize !== undefined);
    }
    ToHTML() {
        // クラスリスト
        let classList = [];
        if (this.Bold) {
            classList.push("b");
        }
        if (this.Italic) {
            classList.push("i");
        }
        if (this.TextDecoration) {
            classList.push(this.TextDecoration);
        }
        if (this.FontSize) {
            classList.push(this.FontSize);
        }
        if (this.Color) {
            classList.push(this.Color);
        }
        // 子要素は継承クラスを利用する。
        let html = super.ToHTML();
        html = `<span class="${classList.join(" ")}">${html}</span>`;
        return html;
    }
    ToAucString() {
        // タグ名
        let tagList = [];
        if (this.Color) {
            tagList.push(BbsTextTag.ColorMap[this.Color].auc);
        }
        if (this.FontSize) {
            tagList.push(BbsTextTag.FontSizeMap[this.FontSize].auc);
        }
        if (this.Bold) {
            tagList.push("太");
        }
        if (this.Italic) {
            tagList.push("斜");
        }
        if (this.TextDecoration) {
            tagList.push(BbsTextTag.TextDecorationMap[this.TextDecoration]);
        }
        let text = super.ToAucString();
        text = `<${tagList.join("")}:${text}>`;
        return text;
    }
    SetFromIAucText(obj) {
        const tag = obj.value.shift();
        if (tag) {
            const tagString = tag.value;
            // 太
            if (tagString.indexOf("太") >= 0) {
                this.Bold = true;
            }
            // 斜め
            if (tagString.indexOf("斜") >= 0) {
                this.Italic = true;
            }
            // 消上下
            for (const key in BbsTextTag.TextDecorationMap) {
                const decoration = BbsTextTag.TextDecorationMap[key];
                if (tagString.indexOf(decoration) >= 0) {
                    this.TextDecoration = key;
                }
            }
            // 大小
            for (const key in BbsTextTag.FontSizeMap) {
                const fontSize = BbsTextTag.FontSizeMap[key].auc;
                if (tagString.indexOf(fontSize) >= 0) {
                    this.FontSize = key;
                }
            }
            // 色
            for (const key in BbsTextTag.ColorMap) {
                const color = BbsTextTag.ColorMap[key].auc;
                if (tagString.indexOf(color) >= 0) {
                    this.Color = key;
                }
            }
        }
        super.SetFromIAucText(obj);
    }
}
/**
 * TextDecorationの型指定や入力処理に使用するデータ
 */
BbsTextTag.TextDecorationMap = {
    d: "消",
    o: "上",
    u: "下",
};
/**
 * FontSizeの型指定や入力処理に使用するデータ
 */
BbsTextTag.FontSizeMap = {
    fs_small: { value: "10px", auc: "小" },
    fs_large: { value: "22px", auc: "大" },
};
/**
 * Colorの型指定や入力処理に使用するデータ
 */
BbsTextTag.ColorMap = {
    c_black: { value: "#000000", auc: "黒" },
    c_white: { value: "#ffffff", auc: "白" },
    c_red: { value: "#fd3838", auc: "赤" },
    c_green: { value: "#28b731", auc: "緑" },
    c_blue: { value: "#5e7dff", auc: "青" },
    c_yellow: { value: "#ffc513", auc: "黄" },
    c_cyan: { value: "#4bd9ea", auc: "水" },
    c_purple: { value: "#aa6dca", auc: "紫" },
    c_orange: { value: "#ff7f2b", auc: "橙" },
    c_brown: { value: "#8e4731", auc: "茶" },
    c_gray: { value: "#c7c7c7", auc: "灰" },
    c_pink: { value: "#ff89af", auc: "桃" },
    // font_rainbowは styleのカラー値では探さないので、比較時に絶対に一致しない文字列にしている。
    font_rainbow: { value: "NotStyleColor", auc: "虹" },
};
/**
 * 掲示板の本体部分のを構成するルビ
 */
class BbsTextRuby extends BbsTextParent {
    constructor() {
        super(...arguments);
        /**
         * <rt>の子要素
         */
        this.Rt = new BbsTextParent();
    }
    /**
     * nodeの内部を検索して、BbsTextRubyのプロパティをセットする。
     * @param node
     */
    SetValueFromHtmElement(node) {
        // 解析のために<rt>を外す必要があるので、クローンを作る。
        const clone = node.cloneNode(true);
        const rt = clone.querySelector("rt");
        if (rt) {
            // <rt>を使って、Rtをセットする。
            this.Rt.SetValueFromHtmElement(rt);
            // そのまま継承クラスを使うと、<rt>まで解析されてしまうので、
            // <rt>>を外した後、継承クラスのメソッドを使う。
            rt.remove();
            super.SetValueFromHtmElement(clone);
        }
    }
    /**
     * ルビとして意味のある情報がセットされているか否か
     */
    IsValid() {
        return (this.ChildElement.length > 0 && this.Rt.ChildElement.length > 0);
    }
    ToHTML() {
        let rt = this.Rt.ToHTML();
        let html = super.ToHTML();
        html = `<ruby>${html}<rt>${rt}</rt></ruby>`;
        return html;
    }
    ToAucString() {
        let rt = this.Rt.ToAucString();
        let text = super.ToAucString();
        text = `{ル:${text},${rt}}`;
        return text;
    }
}
/**
 * その他の要素
 * （基本的には使わないが、考慮漏れしているケースがあるかも知れないので、保険として用意している）
 */
class BbsTextOther {
    constructor(element) {
        this.Element = element;
    }
    ToHTML() {
        return this.Element.outerHTML;
    }
    ToAucString() {
        return this.Element.outerHTML;
    }
}
/**
 * nodeを解析し、対応する IBbsTextElementの派生クラスを返す。
 * @param node
 * @returns
 */
function CreateBbsTextElementFromNode(node) {
    if (node instanceof Text) {
        const nodeValue = node.nodeValue;
        if (nodeValue) {
            const text = new BbsTextText();
            text.Text = nodeValue;
            return text;
        }
    }
    else if (node instanceof HTMLBRElement) {
        return new BbsTextBre();
    }
    else if (node instanceof HTMLAnchorElement) {
        /*
            <a>要素のときは、大きく分けて３つの要素がある。

            ケース１：同じ掲示板へのリンク
            <a class="forceid bbs-mes-link">&gt;&gt;1</a>
            これは掲示板作成時に">>1"とした時に作成されるリンクである。

            ケース２：他の掲示板へのリンク
            <a class="forceid bbs-mes-link">bbs:200239:1</a>
            これは掲示板作成時に"<bbs:200239:1>"とした時に作成されるリンクである。
            記事番号が無い場合（"<bbs:200239>"）も同じ。

            ケース３：キャラへのリンク
            <a title="キャラプロフィールを開く" class="forceid">1xkx:1</a>
            これは掲示板作成時に"<id:1xkx:1>"とした時に作成されるリンクである。
            部隊へのリンクの場合（"<id:1xkx>"）も同じ。

            このうちケース１と２はBbsTextAnchorとして格納する。
            ケース３は、BbsTextAnchorSpanとして格納する。
        */
        const a = new BbsTextAnchor();
        a.Text = node.innerText;
        a.BbsMesLink = node.classList.contains("bbs-mes-link");
        return a;
    }
    else if (node instanceof HTMLImageElement) {
        /*
            <img>要素のときは、基本的にダイス（さいころ）の画像しかない。
            （添付画像は、 <p class="wordBreak">内には挿入されない。別の場所に挿入される）
        */
        const dice = new BbsDice();
        const isValid = dice.SetValueFromHTMLElement(node);
        if (isValid) {
            const textDice = new BbsTextDice();
            textDice.AddDice(dice);
            return textDice;
        }
        else {
            return new BbsTextOther(node);
        }
    }
    else if (node instanceof HTMLSpanElement) {
        // span要素は様々なケースがあるので、専用メソッドで処理する
        return CreateBbsTextElementFromSpan(node);
    }
    else if (node instanceof HTMLElement) {
        if (node.tagName == "RUBY") {
            const ruby = new BbsTextRuby();
            ruby.SetValueFromHtmElement(node);
            return ruby;
        }
        // 本来、ここまで来てはいけないのだが、
        // いずれにも該当しない場合は、Otherを返しておく。
        return new BbsTextOther(node);
    }
    return null;
}
/**
 * spanを解析し、対応する IBbsTextElementの派生クラスを返す。
 * @param span
 * @returns
 */
function CreateBbsTextElementFromSpan(span) {
    /*
    span要素は、大きく分けて
     １．ゲーム内で作成されるケース
     ２．本ツールが作成するケース
    の２種類がある。そして、さらに細かく分類できる。

    １－１：装飾タグに対して作成されるspan
    <span style="rgb(0, 0, 0);">黒　中　</span>
    <span class="b d" style="font-weight: bold; text-decoration: line-through; color: rgb(255, 255, 255); font-size: 22px;">白　大　太　消</span>
    <span class="i o" style="font-style: italic; text-decoration: overline; color: rgb(253, 56, 56); font-size: 10px; line-height: 1.2em;">赤　小　斜　上</span>
    <span class="b i u" style="font-weight: bold; font-style: italic; text-decoration: underline; color: rgb(40, 183, 49);">緑　中　太斜　下</span>
    <span style="color: rgb(94, 125, 255);">青</span>
    <span class="font_rainbow">虹</span>
    <赤　小　斜　上: ××>のような装飾タグに対して作成されるケース。
    このケースでは太斜消上下に対しては、b, i, d, o, u のクラス名が付加される。
    色と大小に対してはクラス名は付加されず、styleの color, font-sizeで判断するしかない。

    １－２：ダイス（さいころ）を使ったときに、２つのダイスの間に作成されるspan
    <span>&nbsp;</span>
    という空白文字１つ分のspan作成される。クラス指定もない。
    このケースは、本来は存在しない要素だが、コピー＆ペースト時に何故か作成されるので、ここで捨ててしまう。
    （保存データに出力してしまうと、ゲーム内の掲示板と比べて余分な空白が存在することになり、表示も一致しなくなる。）

    ２－１：装飾タグに対して本ツールが作成するspan
    <span class="i o fs_small c_red" style="font-style: italic; text-decoration: overline; font-size: 10px; color: rgb(253, 56, 56);">赤　小　斜　上</span>
    のように色や大小に対してクラスの指定されるが、同時に１-１と同じstyleによる指定も存在するので、１-１と同様に処理すればよい。

    ２－２：ダイス（さいころ）に対する代替span
    <span class="sai" data-dicetype="d10" data-value="05">5</span> (styleは省略）
    のように、ダイスに対して本ツールが作成する代替span。
    class="sai"で判断する。

    ２－３：リンクに対する代替span
    <span class="forceid bbs-mes-link">bbs:200239:1</span> (styleは省略）
    のように、<span class="forceid>に対してツールが作成する代替span。
    class="forceid"で判断する。
*/
    // ２－２のダイスの代替spanをチェックする
    if (span.classList.contains("sai")) {
        const dice = new BbsDice();
        const isValid = dice.SetValueFromHTMLElement(span);
        if (isValid) {
            const textDice = new BbsTextDice();
            textDice.AddDice(dice);
            return textDice;
        }
    }
    // ２－３のリンクの代替spanをチェックする
    if (span.classList.contains("forceid")) {
        const a = new BbsTextAnchor();
        a.Text = span.innerText;
        a.BbsMesLink = span.classList.contains("bbs-mes-link");
        return a;
    }
    // １－２をチェックする
    if (span.innerHTML == "&nbsp;" && span.className == "") {
        // このケースは、削除
        return null;
    }
    // １－１をチェックする
    const tag = new BbsTextTag();
    tag.SetValueFromHtmElement(span);
    if (tag.IsValid()) {
        return tag;
    }
    // 本来、ここまで来てはいけないのだが、
    // いずれにも該当しない場合は、Otherを返しておく。
    return new BbsTextOther(span);
}
/**
 * 掲示板の本文部分の情報を保持する。
 */
class BbsText extends BbsTextParent {
    constructor() {
        super(...arguments);
        /**
         * 等倍フォントでの書きこんでいるか否か
         */
        this._Mono = false;
    }
    /**
     * HTMLElementの内部を検索して、BbsTextのプロパティをセットする。
     * @param element
     * @returns
     */
    SetValueFromHtmElement(element) {
        /*
            ゲーム画面をコピペした時
            本文は２種類のフォーマットが存在する。
            
            フォーマット１：等倍フォント指定が無いとき
            <td class="td_b2">
                <p class="wordBreak">
                    本文のデータ
                </p>
            </td>
            
            フォーマット２：等倍フォント指定があるとき
            <td class="td_b2">
                <p class="wordBreak">
                    <span class="mono">
                        本文のデータ
                    </span>
                </p>
            </td>

            どちらのケースも、<td class="td_b2"><p class="wordBreak">までは存在するので、
            その中に<span class="mono">があるか否かでフォーマット１と２を区別する。
        */
        const p_wordBreak = element.querySelector(".td_b2>.wordBreak");
        if (p_wordBreak instanceof HTMLParagraphElement) {
            const span_mono = p_wordBreak.querySelector(".mono");
            if (span_mono instanceof HTMLSpanElement) {
                // フォーマット２（等倍フォント）
                this._Mono = true;
                super.SetValueFromHtmElement(span_mono);
            }
            else {
                // フォーマット１（等倍フォントでない）
                this._Mono = false;
                super.SetValueFromHtmElement(p_wordBreak);
            }
        }
    }
    ToHTML() {
        let html = super.ToHTML();
        if (this._Mono) {
            html = `<span class="mono">${html}</span>`;
        }
        html = `<p class="wordBreak">${html}</p>`;
        return html;
    }
    ToAucString() {
        let text = "";
        if (this._Mono) {
            text += "<等倍フォント>\n";
        }
        text += super.ToAucString();
        return text;
    }
    /**
     * JSON.perse()で使用するメソッド
     *
     * 書き込み本文の情報は、ToAucString()でテキスト化する。
     * @returns
     */
    toJSON() {
        return this.ToAucString();
    }
    SetFromAucString(text) {
        const check = text.indexOf("<等倍フォント>\n");
        if (check == 0) {
            this._Mono = true;
            text = text.slice("<等倍フォント>\n".length);
        }
        super.SetFromAucString(text);
    }
    /**
     * JSON,parseで作られたtextから、BbsTextのプロパティをセットする。
     * @param text
     */
    SetFromJson(text) {
        this.SetFromAucString(text);
    }
}
/**
 * 掲示板の書き込み１件分の情報を保持するクラス
 */
class Bbs {
    constructor() {
        /**
         * 書き込みの本文
         */
        this.Text = new BbsText();
    }
    /**
     * HTMLElementの内を検索して、BBSのプロパティをセットする。
     * @param element 掲示板の１人分の情報を保持したElement
     * @returns
     */
    SetValueFromHtmElement(element) {
        // 発言者アイコン
        // 元データでは、<td class="td_a">内の<img>
        const td_a_img = element.querySelector(".td_a>img");
        if (td_a_img instanceof HTMLImageElement) {
            const img = new BbsIconImg(this);
            img.SetValueFromHtmElement(td_a_img);
            if (img.IsValid()) {
                this.Icon = img;
            }
        }
        // レス番号
        // 元データでは、<span class="res-no">に、`${ResNo}： `というフォーマットで入っている。
        const res_no = element.querySelector(".res-no");
        if (res_no instanceof HTMLElement) {
            const match = res_no.innerText.match(/(\d+)：/);
            if (match) {
                this.ResNo = match[1];
            }
        }
        // 発言者
        // 元データでは、<span class="bbs-force">に入っている。
        const bbs_force = element.querySelector(".bbs-force");
        if (bbs_force instanceof HTMLElement) {
            this.Author = bbs_force.innerText;
        }
        // 発言者の部隊ID
        // 元データでは、<span class="forceid">に入っている。
        const force_id = element.querySelector(".forceid");
        if (force_id instanceof HTMLElement) {
            this.Id = force_id.innerText;
        }
        // 書き込み日時
        // 元データでは、<span class="txt"><span>2022/11/21(月) 01:00:30</span></span>
        // となっている。
        const txt = element.querySelector(".txt");
        if (txt instanceof HTMLElement) {
            this.Date = txt.innerText;
        }
        // 添付された画像
        const img = new BbsAttachedImg();
        img.SetValueFromHtmElement(element);
        if (img.IsValid()) {
            this.AttachedImg = img;
        }
        // 設定項目の「ダイス」
        const dice = new BbsAttachedDice();
        dice.SetValueFromHtmElement(element);
        if (dice.IsValid()) {
            this.AttachedDice = dice;
        }
        // アンケート
        const enquete = new BbsEnquete();
        enquete.SetValueFromHtmElement(element);
        if (enquete.IsValid()) {
            this.Enquete = enquete;
        }
        // 書き込み本文
        this.Text.SetValueFromHtmElement(element);
    }
    /**
     * テンプレートを使って、掲示板情報のHTML要素を作成する。
     *
     * SaveBbs.html内に存在する、tamplatee_idと同じID名のテンプレートを使って
     * 掲示板情報が作成される。
     *
     * @param template_id
     * @returns
     */
    CreateElementByTemplate(template_id) {
        const clone = CloneTemplate(template_id);
        if (clone instanceof HTMLElement) {
            if (this.ResNo) {
                clone.id = Bbs.IdNameForResNo(this.ResNo);
                const res_no = clone.querySelector(".res-no");
                res_no.innerHTML = `${this.ResNo}：&nbsp;`;
            }
            const td_a = clone.querySelector(".td_a");
            if (BbsParameter.IconImg) {
                if (this.Icon) {
                    td_a.innerHTML = this.Icon.ToHTML();
                }
            }
            else {
                td_a.remove();
            }
            if (this.Author) {
                const bbs_force = clone.querySelector(".bbs-force");
                bbs_force.innerText = this.Author;
            }
            if (this.Id && BbsParameter.ForceId) {
                const force_id = clone.querySelector(".forceid");
                force_id.innerText = this.Id;
            }
            if (this.Date && BbsParameter.Date) {
                const txt = clone.querySelector(".txt");
                txt.innerText = this.Date;
            }
            const td_b2 = clone.querySelector(".td_b2");
            let innerHTML = "";
            if (this.AttachedImg && BbsParameter.AttachedImg) {
                innerHTML += this.AttachedImg.ToHTML();
            }
            if (this.AttachedDice) {
                innerHTML += this.AttachedDice.ToHTML();
            }
            innerHTML += this.Text.ToHTML();
            if (this.Enquete) {
                innerHTML += this.Enquete.ToHTML();
            }
            td_b2.innerHTML = innerHTML;
        }
        return clone;
    }
    /**
     * AUCに書き込む時と同じような文字列を作成する。
     */
    ToAucString() {
        let text = "";
        text += `${this.ResNo}:${this.Author}:${this.Id}:${this.Date}\n`;
        if (this.AttachedImg) {
            text += `添付画像:${this.AttachedImg.org}\n`;
        }
        if (this.AttachedDice) {
            text += `添付ダイス:${this.AttachedDice.ToAucString()}\n`;
        }
        text += `\n${this.Text.ToAucString()}\n\n`;
        if (this.Enquete) {
            text += this.Enquete.ToAucString();
        }
        return text;
    }
    /**
     * JSON,parseで作られたobjectからBbsのプロパティをセットする。
     * @param obj
     */
    SetFromJson(obj) {
        if (obj.Icon) {
            const icon = new BbsIconImg(this);
            icon.SetFromJson(obj.Icon);
            obj.Icon = icon;
        }
        Object.assign(this, obj);
    }
    /**
     * レス番号による<a>リンクを張るためにID名を作成する。
     *
     * 名前を付ける側（飛ばれる側）と<a>要素でリンク張る側で、綴りミスが起きないようにメソッド化している。
     *
     * @param resNo レス番号
     */
    static IdNameForResNo(resNo) {
        return `res-no-${resNo}`;
    }
}
/**
 * 色の文字を#RRGGBB形式にして返す。
 * @param color 色の文字
 * @returns
 */
function Color(color) {
    // 余分な空白文字を削除
    color = color.replace(/\s+/g, "");
    // ＃RRGGBB形式の場合をチェック
    let match = color.match(/#[0-9A-F]{6}/i);
    if (match) {
        "#RRGGBB形式だったなら、そのまま返す。";
        return match[0];
    }
    // RGB(r,g,b)形式の場合をチェック
    match = color.match(/RGB\((\d+),(\d+),(\d+)\)/i);
    if (match) {
        let r = ('0' + parseInt(match[1]).toString(16)).slice(-2);
        let g = ('0' + parseInt(match[2]).toString(16)).slice(-2);
        let b = ('0' + parseInt(match[3]).toString(16)).slice(-2);
        return `#${r}${g}${b}`;
    }
    // どちらのケースにもマッチしないなら""を返す。
    return "";
}
/**
 * idの<template>のcloneを作り返す。
 * @param id テンプレートのID名
 * @returns cloneしたHTML要素
 */
function CloneTemplate(id) {
    const template = document.getElementById(id);
    if (template instanceof HTMLTemplateElement) {
        if (template.content.firstElementChild) {
            return template.content.firstElementChild.cloneNode(true);
        }
    }
}
/**
 * 掲示板のタイトル・親記事情報を保持するクラス
 */
class BbsParent extends Bbs {
    constructor() {
        // 日時や親記事など、class Bbsとほぼ同じ情報があるので、Bbsを継承する。
        // （違いは１点。レス番号が付かないことだけ。）
        super(...arguments);
        // その他に親記事特有の情報をして、以下のプロパティがある。
        /**
         * 掲示板のタイトル
         */
        this.Title = "";
        /**
         * 掲示板番号
         */
        this.BbsNo = "";
    }
    /**
     * HTMLElementの内を検索して、BbsTitleのプロパティをセットする。
     * @param element 掲示板の１人分の情報を保持したElement
     * @returns
     */
    SetValueFromHtmElement(element) {
        // 掲示板タイトル
        // 元データでは、<span class="child-title">にある
        const child_title = element.querySelector(".child-title");
        if (child_title instanceof HTMLElement) {
            this.Title = child_title.innerText;
        }
        // 掲示板番号
        // 元データでは、<div class="bbs-number">に`掲示板No.${bbsNo}`
        // というフォーマットの6桁の数値で入っている。
        const bbs_number = element.querySelector(".bbs-number");
        if (bbs_number instanceof HTMLElement) {
            const match = bbs_number.innerText.match(/No\.(\d{6})/);
            if (match) {
                this.BbsNo = match[1];
            }
        }
        // 親記事に本文や日時などは、他の掲示板の書き込みと同じ形式になっているので、
        // 継承元クラスのメソッドでセットすることができる。
        super.SetValueFromHtmElement(element);
    }
    /**
     * テンプレートを使って、掲示板の親記事を作成する。
     * @param template_id
     * @returns
     */
    CreateElementByTemplate(template_id) {
        // 掲示板の親記事部分を作る
        const bbs_con = super.CreateElementByTemplate(template_id);
        if (bbs_con) {
            const clone = CloneTemplate('title');
            if (clone instanceof HTMLElement) {
                const child_title = clone.querySelector(".child-title");
                child_title.innerText = this.Title;
                const bbs_number = clone.querySelector(".bbs-number");
                bbs_number.innerText = `掲示板No.${this.BbsNo}`;
                clone.appendChild(bbs_con);
                return clone;
            }
        }
    }
    ToAucString() {
        let text = `BBSID:${this.BbsNo}:${this.Title}\n`;
        text += "ROOT" + super.ToAucString();
        return text;
    }
    /**
     * formatに指定された文字列中の、以下の文字列をプロパティ値に置き換えた文字を作る。
     *
     * %T => Title
     *
     * %N => BbsNo
     *
     * 例： this.Sprintf("aucbbs%N") は \`abcbbs${this.BbsNo}\`と同じになる。
     * @returns
     */
    Sprintf(format) {
        let value = format.replace(/%[%TN]/g, (substring) => {
            if (substring == '%T') {
                return this.Title;
            }
            else if (substring == '%N') {
                return this.BbsNo;
            }
            return '%';
        });
        return value;
    }
}
/**
 * Bbsの１つのスレッドを管理するクラス
 */
class BbsThread {
    constructor() {
        /**
        * 掲示板の子記事のリスト
        */
        this.Children = [];
    }
    /**
     * Bbsを登録する。
     * ただし、resNoのないデータは登録しない
     *
     * 登録済みデータの中にresNoが重複するものがある場合は、上書きする。（以前のものは廃棄される）
     * @param bbs
     */
    SetBbs(bbs) {
        const resNo = Number(bbs.ResNo);
        if (resNo > 0) {
            this.Children[resNo] = bbs;
        }
    }
    /**
     * 全データを削除する。
     */
    Clear() {
        this.Parent = undefined;
        this.Children = [];
    }
    /**
     * forEach
     * @param func
     */
    forEach(func) {
        let index = 0;
        this.Children.forEach(bbs => {
            func(bbs, index, this.Children);
            index++;
        });
    }
    /**
     * Listの長さ（＝登録している掲示板の数）を返す。
     */
    get Length() { return this.Children.length; }
    /**
     * JSON.perse()で使用するメソッド
     *
     * @returns
     */
    toJSON() {
        // this.Childrenは、そのままだとnullになっている部分があるので、
        // nullでないものだけのリストに置き換える。
        const list = this.Children.filter(bbs => (bbs != undefined));
        const obj = {
            Parent: this.Parent,
            Children: list,
        };
        return obj;
    }
}
/**
 * ビューアー部分(\<div class="viewer">)に存在するHTML要素に関する処理をまとめたクラス
 */
class Viewer {
    constructor() {
        /**
         * Viewerに表示されている画面
         */
        this._select = 'information';
    }
    /**
     * \<div id="bbs_viewer">に対するHTMLDivElement
     */
    get BbsViewer() {
        return this.bbs_viewer;
    }
    ;
    /**
     * \<pre id="html_data">に対するHTMLPreElement
     */
    get HtmlData() {
        return this.html_data;
    }
    /**
     * 初期化処理
     */
    Init() {
        this.information = document.getElementById('information');
        this.bbs_viewer = document.getElementById('bbs_viewer');
        this.html_data = document.getElementById('html_data');
        this.Select = 'information';
    }
    /**
     * Viewerに表示されている画面
     */
    get Select() { return this._select; }
    set Select(value) {
        this._select = value;
        // informationの表示／非表示を変更
        if (this.information) {
            this.information.style.display = this._select == 'information' ? "" : "none";
        }
        // bbs_viewerの表示／非表示を変更
        if (this.bbs_viewer) {
            this.bbs_viewer.style.display = this._select == 'bbs_viewer' ? "" : "none";
        }
        // html_dataの表示／非表示を変更
        if (this.html_data) {
            this.html_data.style.display = this._select == 'html_data' ? "" : "none";
        }
    }
}
/**
 * メニューに存在するのHTML要素に関する処理をまとめたクラス
 */
class Menu {
    constructor() {
        /**
         * button 要素のマップ
         */
        this.Btn = new Map();
        /**
         * input check 要素のマップ
         */
        this.CheckElement = new Map();
        /**
         * その他の input要素のマップ
         */
        this.InputElement = new Map();
        /**
         * select要素のマップ
         */
        this.SelectElement = new Map();
        /**
         * オブザーバーリスト
         */
        this.observer_list = [];
    }
    /**
     * オブザーバーを登録する。
     * @param observer
     */
    AddObserver(observer) {
        this.observer_list.push(observer);
    }
    /**
     * オブザーバーに通知する。
     * @param message
     */
    NortfyObserver(message) {
        this.observer_list.forEach(obs => obs.Nortify(message));
    }
    /**
     * 初期化処理
     */
    Init() {
        this.bbs_title = document.getElementById("bbs_title");
        // bbs_titleのイベントハンドラ
        if (this.bbs_title) {
            this.bbs_title.addEventListener('change', () => {
                this.NortfyObserver('bbs_title');
            });
        }
        // Mapとイベントハンドラを登録する
        // ボタン型のclickハンドラを登録する。
        Menu.BtnId.forEach(id => {
            const btn = document.getElementById(id);
            if (btn instanceof HTMLButtonElement) {
                this.Btn.set(id, btn);
                btn.addEventListener('click', () => {
                    this.NortfyObserver(id);
                });
            }
        });
        //input check型のchangeハンドラを登録する。
        Menu.CheckId.forEach(id => {
            const element = document.getElementById(id);
            if (element instanceof HTMLInputElement) {
                this.CheckElement.set(id, element);
                element.addEventListener('change', () => {
                    this.NortfyObserver(id);
                });
            }
        });
        // check以外のinput型のchangeハンドラを登録する。
        Menu.InputId.forEach(id => {
            const element = document.getElementById(id);
            if (element instanceof HTMLInputElement) {
                this.InputElement.set(id, element);
                element.addEventListener('change', () => {
                    if (element.pattern) {
                        const re = new RegExp(element.pattern);
                        const match = element.value.match(re);
                        if (match) {
                            if (match[0] != element.value) {
                                element.value = match[0];
                            }
                        }
                        else {
                            element.value = "";
                        }
                    }
                    this.NortfyObserver(id);
                });
            }
        });
        //select型のchangeハンドラを登録する。
        Menu.SelectId.forEach(id => {
            const element = document.getElementById(id);
            if (element instanceof HTMLSelectElement) {
                this.SelectElement.set(id, element);
                element.addEventListener('change', () => {
                    this.NortfyObserver(id);
                });
            }
        });
    }
    /**
     * input check 型のメニューの値を得る
     * @param id
     * @returns
     */
    Check(id) {
        const element = this.CheckElement.get(id);
        return element ? element.checked : false;
    }
    /**
     * select型の選択さえている値を得る
     * @param id
     */
    Select(id) {
        const element = this.SelectElement.get(id);
        return element ? element.value : "";
    }
    /**
     * 掲示板のタイトル
     * @param value
     */
    set BbsTitle(value) {
        if (this.bbs_title) {
            this.bbs_title.value = value;
        }
    }
    get BbsTitle() {
        return this.bbs_title ? this.bbs_title.value : "";
    }
    /**
     * 生成するHTML中の掲示板タイトル
     */
    get BbsTitleForCreateBbsHtml() {
        return this.GetValueOrPlaceholder(this.bbs_title);
    }
    /**
     * placeholderを考慮して、elementの値を得る。
     * @param elemnet
     */
    GetValueOrPlaceholder(element) {
        if (element) {
            if (element.value) {
                return element.value;
            }
            else {
                return element.placeholder;
            }
        }
        return "";
    }
    /**
     * HTMLファイル名
     */
    set Htmlname(value) {
        const element = this.InputElement.get('htmlname');
        if (element) {
            element.value = value;
        }
    }
    get Htmlname() {
        const element = this.InputElement.get('htmlname');
        return element ? element.value : "";
    }
    /**
     * ダウンロードするHTMLのファイル名
     */
    get HtmlnameForDownload() {
        return this.GetValueOrPlaceholder(this.InputElement.get('htmlname')) + ".html";
    }
    /**
     * 画像を保存するフォルダ名
     */
    get ImageFolderName() {
        return this.GetValueOrPlaceholder(this.InputElement.get('image_folder'));
    }
}
/**
 * メニューの button 要素のID名
 */
Menu.BtnId = [
    'btn_change_view',
    'btn_clear',
    'btn_dl_css',
    'btn_dl_html',
    'btn_paste',
    'btn_dl_json',
    'btn_up_json',
];
/**
 * メニューの check 要素のID名
 */
Menu.CheckId = [
    'check_attached_img',
    'check_bbs_board',
    'check_date',
    'check_force_id',
    'check_grid_format',
    'check_icon_img',
    'check_image_save_mode',
];
/**
 * その他のinput要素のID名
 */
Menu.InputId = [
    'image_folder',
    'htmlname',
];
/**
 * メニューの select 要素のID名
 */
Menu.SelectId = [
    'select_view',
];
const menu = new Menu();
const viewer = new Viewer();
const bbsThread = new BbsThread();
function SaveBbs() {
    // メニュー初期化
    menu.Init();
    // Actionを初期化し、メニューと結びつける
    const action = new Action();
    menu.AddObserver(action);
    // Viewer初期化
    viewer.Init();
    // 戻るボタンでページを戻ってきたときのイベントハンドラを登録する
    window.addEventListener('pageshow', (e) => {
        if (e.persisted) {
            // 残っているデータをクリアする
            Clear();
        }
    });
}
/**
 * メニューの変化に応じて、さまざまにアクションする為のクラス
 */
class Action {
    Nortify(message) {
        switch (message) {
            case 'btn_paste':
                Paste();
                break;
            case 'btn_clear':
                Clear();
                break;
            case 'btn_change_view':
                ChangeViewer();
                break;
            case 'btn_dl_html':
                DownloadTextData(CreateBbsHtml(), menu.HtmlnameForDownload);
                break;
            case 'btn_dl_css':
                Download("aucbbs.css", "aucbbs.css");
                break;
            case 'btn_dl_json':
                const jsonFilename = menu.HtmlnameForDownload.replace(".html", ".json");
                DownloadTextData(CreateJSON(), jsonFilename);
                break;
            case 'btn_up_json':
                ReadJson();
                break;
            case 'check_bbs_board':
            case 'check_icon_img':
            case 'check_attached_img':
            case 'check_force_id':
            case 'check_date':
            case 'check_grid_format':
            case 'check_image_save_mode':
            case 'image_folder':
                UpdateBbsViwer();
                UpdateHtmlData();
                break;
            case 'bbs_title':
            case 'select_view':
                UpdateHtmlData();
                break;
        }
    }
}
/**
 * Viewerに表示されている画面を切り替える。
 */
function ChangeViewer() {
    // 現在値に応じて、切り替える
    switch (viewer.Select) {
        case 'bbs_viewer':
            // bbs_viewerを表示中ならhtml_dataに切り替える。
            viewer.Select = 'html_data';
            break;
        case 'html_data':
            // html_dataを表示中ならbbs_viwerに切り替える。
            viewer.Select = 'bbs_viewer';
            break;
        default:
            // それ以外は変更しない
            break;
    }
}
/**
 * クリックボードからペーストする処理
 */
async function Paste() {
    await ReadBbsDataFromClipboard();
    // ビューの更新
    UpdateBbsViwer();
    UpdateHtmlData();
    // 掲示板データがある場合はビューをチェンジ
    if (bbsThread.Length > 0) {
        viewer.Select = 'bbs_viewer';
    }
}
/**
 * クリップボードから'text/html'を読みとり、英雄クロニクルの掲示板の情報として解析する。
 *
 * 解析の結果掲示板の情報を獲得できれば、それをリストにして返す。
 * @returns
 */
async function ReadBbsDataFromClipboard() {
    try {
        const items = await navigator.clipboard.read();
        for (const item of items) {
            if (!item.types.includes('text/html')) {
                continue;
            }
            const blob = await item.getType('text/html');
            let html = await blob.text();
            const perser = new DOMParser();
            const doc = perser.parseFromString(html, 'text/html');
            // 圧縮されたURLを展開するためのデータがあれば読みとる。
            BbsParameter.UrlConverter.SetValueFromHtmElement(doc);
            // 掲示板のタイトル・親記事情報は、<div class="bbs-board">にあるので、
            // それを取得し、解析する。
            const bbs_board = doc.querySelector(".bbs-board");
            if (bbs_board instanceof HTMLElement) {
                const bbsParent = new BbsParent();
                bbsParent.SetValueFromHtmElement(bbs_board);
                bbsThread.Parent = bbsParent;
                if (menu.BbsTitle == "") {
                    menu.BbsTitle = bbsParent.Sprintf('%T [掲示板No.%N]');
                }
                if (menu.Htmlname == "") {
                    menu.Htmlname = bbsParent.Sprintf('aucbbs%N');
                }
            }
            // 掲示板の１人分の書き込み情報は<table class="bbs-con_ta">にあるので、
            // そのリストを作成
            const bbs_list = doc.querySelectorAll(".bbs-con_ta");
            for (let bbs of bbs_list) {
                // 先祖に.bbs-boardがある場合は、親記事の部分になるので、スキップする。
                if (bbs.closest(".bbs-board")) {
                    continue;
                }
                // HTML要素を解析して、bbsTThreadに情報を追加していく。
                if (bbs instanceof HTMLElement) {
                    const bbsData = new Bbs();
                    bbsData.SetValueFromHtmElement(bbs);
                    bbsThread.SetBbs(bbsData);
                }
            }
        }
    }
    catch (error) {
        console.error(error.message);
    }
}
/**
 * <div id="bbs_viewer">を更新する
 */
function UpdateBbsViwer() {
    // 掲示板作成時に使用するテンプレートを選択する
    const template_id = menu.Check('check_grid_format') ? 'grid-type' : 'table-type';
    const bbs_viewer = viewer.BbsViewer;
    bbs_viewer.innerHTML = "";
    BbsParameter.ImageSaveMode = menu.Check('check_image_save_mode');
    BbsParameter.IconImg = menu.Check('check_icon_img');
    BbsParameter.AttachedImg = menu.Check('check_attached_img');
    BbsParameter.ForceId = menu.Check('check_force_id');
    BbsParameter.Date = menu.Check('check_date');
    BbsParameter.ImgFolder = menu.ImageFolderName;
    BbsParameter.UrlConverter.Init();
    // 親記事部分をbbs_viewerに追加する。
    if (menu.Check('check_bbs_board') && bbsThread.Parent) {
        const bbs_board = bbsThread.Parent.CreateElementByTemplate(template_id);
        if (bbs_board) {
            bbs_viewer.appendChild(bbs_board);
        }
    }
    // 普通の記事をbbs_viewerに追加する。
    bbsThread.forEach(bbsData => {
        const bbs_element = bbsData.CreateElementByTemplate(template_id);
        if (bbs_element) {
            bbs_viewer.appendChild(bbs_element);
        }
    });
    // 圧縮されたURLを展開するためのデータを追加する。
    bbs_viewer.insertAdjacentHTML("beforeend", BbsParameter.UrlConverter.ToHTML());
}
/**
 * <pre id="html_data">を更新する
 */
function UpdateHtmlData() {
    switch (menu.Select("select_view")) {
        case "HTML":
            viewer.HtmlData.innerText = CreateBbsHtml();
            break;
        case "AucString":
            viewer.HtmlData.innerText = CreateAucText();
            break;
        case "JSON":
            viewer.HtmlData.innerText = CreateJSON();
            break;
    }
}
/**
 * AUCの掲示板に書き込むときと同様のテキストを作成する。
 * 本文以外の要素（レス番号、添付画像、アンケートなど）も、参考としてテキスト化している。
 *
 * また、ダイスはすでに出目が確定しているので、本来のAUCのさいころタグとは異なっている。
 * @returns
 */
function CreateAucText() {
    let text = "";
    text += "#ＡＵＣ掲示板に書き込む時のデータ\n";
    text += "# 本文以外の要素（レス番号、添付画像、アンケートなど）も、参考として表示しています。\n";
    text += "# ダイスはすでに出目が確定しているので、ＡＵＣの本来のダイスタグとは異なっています。\n";
    text += "# 例：さいころ２個は<dice:2d>だが、出目を追加して<dice:2d:01:03>のようになっている。\n";
    text += "\n";
    if (bbsThread.Parent) {
        text += bbsThread.Parent.ToAucString();
    }
    bbsThread.forEach(bbsData => {
        text += "==========\n";
        text += bbsData.ToAucString();
    });
    return text;
}
/**
 * 画面に表示されている掲示板との同等のHTMLデータを作る。
 * @returns
 */
function CreateBbsHtml() {
    let bbsTitle = menu.BbsTitleForCreateBbsHtml;
    let html = "";
    // ファイルの最初から</header> <body>までをセット
    html = '<!DOCTYPE html>\n'
        + '<html lang="ja">\n'
        + '\n'
        + '<head>\n'
        + '  <meta charset="UTF-8">\n'
        + '  <meta http-equiv="X-UA-Compatible" content="IE=edge">\n'
        + '  <meta name="viewport" content="width=device-width, initial-scale=1.0">\n'
        + `  <meta name="generator" content="AucTools/SaveBbs V${version}">\n`
        + `  <title>${bbsTitle}</title>\n`
        + '  <style>\n'
        + '    @import url(\'https://fonts.googleapis.com/css2?family=Inconsolata:wght@400;700&family=Noto+Sans+JP:wght@400;700&display=swap\');\n'
        + '  </style>\n'
        + '  <link rel="stylesheet" type="text/css" href="aucbbs.css">\n'
        + '</head>\n'
        + '\n'
        + '<body>\n'
        + `  <h1>${bbsTitle}</h1>\n`;
    // <body>の中身を作る。
    const bbs_viewer = viewer.BbsViewer.cloneNode(true);
    if (menu.Check('check_image_save_mode')) {
        for (const img of bbs_viewer.querySelectorAll("img")) {
            img.removeAttribute("src");
        }
    }
    let bbs_root_html = "";
    bbs_root_html += '<div id="bbs_root" class="bbs_container">';
    bbs_root_html += bbs_viewer.innerHTML;
    bbs_root_html += '</div>';
    html += FormatHtml(bbs_root_html, 2) + "\n";
    // </body>をセット
    html += '</body>\n';
    if (menu.Check('check_image_save_mode')) {
        // 画像保存モードのためのスクリプトを追加する
        const script = document.getElementById("scr_image_save_mode");
        if (script) {
            html += script.outerHTML;
        }
    }
    //</html>をセット
    html += '\n</html>\n';
    return html;
}
/**
 * HTMLの整形を行う。
 *
 * 掲示板情報から作成したHTMLは改行やインデントが無い。
 * そのまま出力したものをエディタで確認したり、編集したりすることがとても難しい。
 *
 * そこで、改行やインデントを追加し、ある程度の読みやすい形に整形する。
 * @param html HTMLElement.innerHTML が返すようなHTML形式のテキスト
 * @param indent インデントの初期値（default: 0）
 * @returns 整形後のテキスト
 */
function FormatHtml(html, indent = 0) {
    const indentWidth = 2;
    // いったんすべての<XXXX>と</XXXX>の前後に改行を入れる。
    html = html.replace(/(<[^>]+>)/g, "\n$1\n");
    // 改行と空白が続くところは、改行１文字に変更する。
    html = html.replace(/([ \n]*\n[ \n]*)+/g, "\n");
    // <a>と<span>の後ろの改行、</a>と</span>と<br>の前の改行を消す。
    html = html.replace(/(<(a|span).*>)\n/g, "$1");
    html = html.replace(/\n(<(br|\/a|\/span)>)/g, "$1");
    // インデント調整
    const tmp = [];
    for (let line of html.split(/\n/)) {
        if (!line) {
            continue;
        }
        // </table>や</tbody>など行からインデントが減る
        if (line.match(/^\s*<\/(table|tbody|tr|td|p|div)/)) {
            indent -= indentWidth;
        }
        line = " ".repeat(indent) + line;
        tmp.push(line);
        // <table>や<tbody>など行tの次からインデントが増える
        if (line.match(/^\s*<(table|tbody|tr|td|p|div)/)) {
            indent += indentWidth;
        }
    }
    html = tmp.join("\n");
    return html;
}
/**
 * 掲示板データをクリアする
 */
function Clear() {
    bbsThread.Clear();
    // 掲示板タイトルとHTMLファイル名もクリアする。
    menu.BbsTitle = "";
    menu.Htmlname = "";
    // ビューの更新
    UpdateBbsViwer();
    UpdateHtmlData();
    // 表示するデータがないので、替りにインフォメーションを表示する。
    viewer.Select = 'information';
}
/**
 * urlで指定されるファイルをfilenameという名前でダウンロード（ファイル保存）する。
 * @param url
 * @param filename
 */
function Download(url, filename) {
    const dlLink = document.createElement("a");
    dlLink.href = url;
    dlLink.download = filename;
    document.body.insertAdjacentElement('beforeend', dlLink);
    dlLink.click();
    dlLink.remove();
}
/**
 * textDataの文字列をfilenameという名前でダウンロード（ファイル保存）する。
 * @param textData
 * @param filename
 */
function DownloadTextData(textData, filename) {
    const blob = new Blob([textData], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    Download(url, filename);
}
/**
 * 掲示板データからJSON文字列を作成する。
 * @returns
 */
function CreateJSON() {
    const obj = {
        Generator: `SaveBbs/${version}`,
        BbsThread: bbsThread,
    };
    return JSON.stringify(obj, undefined, 1);
}
/**
 * <input id="file">に指定されたJSONファイルを入力し、掲示板データを復元する。
 */
async function ReadJson() {
    const file = await Upload();
    if (file) {
        let reader = new FileReader();
        const promise = new Promise((resolve, reject) => {
            reader.onload = () => resolve(reader.result);
            reader.onerror = () => reject(new Error(`${file.name}の読み込みに失敗しました。`));
            reader.readAsText(file);
        });
        try {
            const json = await promise;
            const obj = JSON.parse(json, BbsReviver);
            const thread = obj.BbsThread;
            if (thread) {
                bbsThread.Clear();
                if (thread.Parent) {
                    bbsThread.Parent = thread.Parent;
                    menu.BbsTitle = thread.Parent.Sprintf('%T [掲示板No.%N]');
                    menu.Htmlname = thread.Parent.Sprintf('aucbbs%N');
                }
                thread.Children.forEach(bbs => {
                    bbsThread.SetBbs(bbs);
                });
                // ビューの更新
                UpdateBbsViwer();
                UpdateHtmlData();
                // 掲示板データがある場合はビューをチェンジ
                if (bbsThread.Length > 0) {
                    viewer.Select = 'bbs_viewer';
                }
            }
            function BbsReviver(key, value) {
                switch (key) {
                    case "Parent":
                        const parent = new BbsParent();
                        parent.SetFromJson(value);
                        bbsThread.Parent = parent;
                        return parent;
                    case "Children":
                        return value.map((obj) => {
                            const bbs = new Bbs();
                            bbs.SetFromJson(obj);
                            return bbs;
                        });
                    case "AttachedDice":
                        const dice = new BbsAttachedDice();
                        dice.SetFromJson(value);
                        return dice;
                    case "AttachedImg":
                        const img = new BbsAttachedImg();
                        img.SetFromJson(value);
                        return img;
                    case "Enquete":
                        const enquate = new BbsEnquete();
                        enquate.SetFromJson(value);
                        return enquate;
                        break;
                    case "EnquateItems":
                        return value.map((obj) => {
                            const item = new BbsEnqueteItem();
                            item.SetFromJson(obj);
                            return item;
                        });
                        break;
                    case "Text":
                        const bbsText = new BbsText();
                        bbsText.SetFromJson(value);
                        return bbsText;
                    default:
                        return value;
                }
            }
        }
        catch (e) {
            const err = e;
            const message = `JSONの入力に失敗しました。\n${err.message}`;
            alert(message);
        }
    }
}
async function Upload() {
    const file = document.createElement("input");
    file.type = "file";
    const promise = new Promise((resolve) => {
        file.addEventListener("change", () => {
            if (file.files && file.files.length > 0) {
                resolve(file.files[0]);
            }
        });
    });
    file.click();
    return promise;
}
const version = "1.1.0";
/* Version 1.1.0 2024/02/29 */
/* Version 1.0.1 2023/02/19 */
/* Version 1.0.0 2023/02/01 */
/* Version 0.1.0 2023/01/08 */
/* Version 0.0.6 2023/01/04 */
/* Version 0.0.5 2022/12/24 */
/* Version 0.0.4 2022/12/19 */
/* Version 0.0.3 2022/12/10 */ 
